// Global variables
let allEndpoints = [];
let openDetailId = null;
let activeCategory = 'all';
let currentSearchTerm = '';
let currentPage = 1;
const endpointsPerPage = 8;

const statusCodes = [
    { code: 200, desc: "OK - Request successful", icon: "✓", color: "text-green-600" },
    { code: 201, desc: "Created - Resource created successfully", icon: "✓", color: "text-green-600" },
    { code: 204, desc: "No Content - Success without content", icon: "✓", color: "text-green-600" },
    { code: 400, desc: "Bad Request - Invalid parameters or missing fields", icon: "!", color: "text-amber-600" },
    { code: 401, desc: "Unauthorized - API Key required or invalid", icon: "!", color: "text-amber-600" },
    { code: 403, desc: "Forbidden - Access denied", icon: "!", color: "text-amber-600" },
    { code: 404, desc: "Not Found - Resource not found", icon: "?", color: "text-blue-600" },
    { code: 405, desc: "Method Not Allowed - HTTP method not supported", icon: "✗", color: "text-red-600" },
    { code: 422, desc: "Unprocessable Entity - Validation failed", icon: "!", color: "text-amber-600" },
    { code: 429, desc: "Too Many Requests - Rate limit exceeded", icon: "⚠", color: "text-amber-600" },
    { code: 500, desc: "Internal Server Error - Server encountered an error", icon: "✗", color: "text-red-600" },
];

// Utility functions
function formatJSON(json) {
    if (typeof json === 'string') {
        try {
            return JSON.stringify(JSON.parse(json), null, 2);
        } catch {
            return json;
        }
    }
    return JSON.stringify(json, null, 2);
}

function getMethodStyle(method) {
    const methodMap = {
        'GET': { badge: 'method-badge-get', color: '#10b981' },
        'POST': { badge: 'method-badge-post', color: '#3b82f6' },
        'PUT': { badge: 'method-badge-put', color: '#f59e0b' },
        'DELETE': { badge: 'method-badge-delete', color: '#ef4444' },
        'PATCH': { badge: 'method-badge-patch', color: '#8b5cf6' },
        'HEAD': { badge: 'bg-gray-600', color: '#6b7280' },
        'OPTIONS': { badge: 'bg-gray-500', color: '#6b7280' }
    };
    return methodMap[method] || { badge: 'bg-gray-500', color: '#6b7280' };
}

function generateCurlCommand(api, params, url) {
    let curl = `curl -X ${api.method}`;

    if (api.method === 'POST' || api.method === 'PUT' || api.method === 'PATCH') {
        curl += ` \\\n  -H 'Content-Type: application/json'`;
    }

    curl += ` \\\n  '${url}'`;

    if ((api.method === 'POST' || api.method === 'PUT' || api.method === 'PATCH') && Object.keys(params).length > 0) {
        curl += ` \\\n  -d '${JSON.stringify(params)}'`;
    }

    return curl;
}

function copyToClipboard(elementId) {
    const element = document.getElementById(elementId);
    if (element) {
        element.select();
        document.execCommand('copy');

        // Show feedback
        const originalValue = element.value;
        element.value = 'Copied!';
        element.classList.add('copy-success');

        setTimeout(() => {
            element.value = originalValue;
            element.classList.remove('copy-success');
        }, 1500);
    }
}

function animateCounter(element, target) {
    const duration = 800;
    const start = 0;
    const increment = target / (duration / 16);
    let current = start;

    const timer = setInterval(() => {
        current += increment;
        if (current >= target) {
            element.textContent = target;
            clearInterval(timer);
        } else {
            element.textContent = Math.floor(current);
        }
    }, 16);
}

function updateStats() {
    const totalEndpoints = allEndpoints.length;
    const categories = [...new Set(allEndpoints.map(api => api.kategori || 'General'))];
    const filtered = getFilteredEndpoints();
    const filteredCount = filtered.length;

    const totalEl = document.getElementById('total-endpoints');
    const categoriesEl = document.getElementById('total-categories');
    const filteredEl = document.getElementById('filtered-count');

    animateCounter(totalEl, totalEndpoints);
    animateCounter(categoriesEl, categories.length);
    animateCounter(filteredEl, filteredCount);

    document.getElementById('active-category').textContent = activeCategory === 'all' ? 'All' : activeCategory;
    document.getElementById('last-updated').textContent = new Date().toLocaleString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function getFilteredEndpoints() {
    let filtered = allEndpoints;

    if (activeCategory !== 'all') {
        filtered = filtered.filter(api => (api.kategori || 'General') === activeCategory);
    }

    if (currentSearchTerm) {
        const term = currentSearchTerm.toLowerCase();
        filtered = filtered.filter(api =>
            api.nama.toLowerCase().includes(term) ||
            api.deskripsi.toLowerCase().includes(term) ||
            api.endpoint.toLowerCase().includes(term) ||
            (api.tags && api.tags.some(tag => tag.toLowerCase().includes(term)))
        );
    }

    return filtered;
}

function renderCategories() {
    const categoriesNav = document.getElementById('categories-nav');
    const categories = ['all', ...new Set(allEndpoints.map(api => api.kategori || 'General'))];

    categoriesNav.innerHTML = categories.map(category => {
        const count = category === 'all'
            ? allEndpoints.length
            : allEndpoints.filter(api => (api.kategori || 'General') === category).length;

        const isActive = category === activeCategory;

        return `
            <button class="category-filter ${isActive ? 'active-filter' : ''}" 
                    data-category="${category}">
                ${category === 'all' ? 'All' : category} 
                <span class="ml-1 text-xs opacity-80">(${count})</span>
            </button>
        `;
    }).join('');

    document.querySelectorAll('.category-filter').forEach(btn => {
        btn.addEventListener('click', function () {
            const category = this.dataset.category;

            if (category === activeCategory) return;

            activeCategory = category;
            currentPage = 1;

            document.querySelectorAll('.category-filter').forEach(b =>
                b.classList.remove('active-filter')
            );
            this.classList.add('active-filter');

            filterEndpoints(currentSearchTerm);

            if (openDetailId) {
                toggleDetail(openDetailId);
            }

            updateStats();
        });
    });
}

function filterEndpoints(searchTerm = '') {
    currentSearchTerm = searchTerm;
    const filtered = getFilteredEndpoints();

    renderEndpoints(filtered);
    updateSearchCount(filtered.length);
    updateStats();
    updateLoadMoreButton(filtered.length);
}

function updateSearchCount(count) {
    const searchCount = document.getElementById('search-count');
    searchCount.textContent = `${count} result${count !== 1 ? 's' : ''}`;

    if (count === 0 && (currentSearchTerm || activeCategory !== 'all')) {
        document.getElementById('empty-state').classList.remove('hidden');
        document.getElementById('endpoints-container').classList.add('hidden');
    } else {
        document.getElementById('empty-state').classList.add('hidden');
        document.getElementById('endpoints-container').classList.remove('hidden');
    }
}

function updateLoadMoreButton(totalCount) {
    const loadMoreContainer = document.getElementById('load-more-container');
    if (totalCount > endpointsPerPage * currentPage) {
        loadMoreContainer.classList.remove('hidden');
    } else {
        loadMoreContainer.classList.add('hidden');
    }
}

function loadAPIData() {
    const container = document.getElementById('endpoints-container');
    container.innerHTML = `
        <div class="text-center py-12">
            <div class="loading-dots inline-block mb-4">
                <span></span>
                <span></span>
                <span></span>
            </div>
            <p class="text-cyan-400">Loading endpoints...</p>
        </div>
    `;

    const apiEndpoints = ['/api/info', '/api/endpoints', '/api/docs'];

    const tryLoad = (index) => {
        if (index >= apiEndpoints.length) {
            showError();
            return;
        }

        fetch(apiEndpoints[index])
            .then(async res => {
                if (!res.ok) throw new Error(`HTTP ${res.status}`);
                return res.json();
            })
            .then(data => {
                allEndpoints = data.apis || data.endpoints || data || [];
                renderCategories();
                filterEndpoints();

                const banner = document.getElementById('api-banner');
                if (banner && banner.complete && banner.naturalWidth > 0) {
                    banner.style.opacity = '0';
                    setTimeout(() => {
                        banner.style.transition = 'opacity 0.6s ease';
                        banner.style.opacity = '1';
                    }, 100);
                }
            })
            .catch(err => {
                console.log(`Trying next endpoint, ${apiEndpoints[index]} failed:`, err);
                tryLoad(index + 1);
            });
    };

    tryLoad(0);
}

function showError() {
    const container = document.getElementById('endpoints-container');
    container.innerHTML = `
        <div class="text-center py-12 bg-gray-900 rounded-xl border border-red-400">
            <div class="w-12 h-12 mx-auto mb-4 rounded-full bg-red-400/10 flex items-center justify-center border border-red-400">
                <svg class="w-6 h-6 text-red-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
            </div>
            <h3 class="text-lg font-semibold text-red-400 mb-2">Failed to Load Data</h3>
            <p class="text-gray-400 text-sm mb-4">Unable to connect to API server. Check your connection.</p>
            <button onclick="loadAPIData()" 
                    class="px-4 py-2 rounded-lg text-sm font-medium transition-colors border border-cyan-400 text-cyan-400 hover:border-cyan-300">
                Try Again
            </button>
        </div>
    `;
}

function toggleDetail(id) {
    const detail = document.getElementById(`detail-panel-${id}`);
    const icon = document.getElementById(`toggle-icon-${id}`);
    const card = document.getElementById(`endpoint-${id}`);

    if (openDetailId && openDetailId !== id) {
        const prevDetail = document.getElementById(`detail-panel-${openDetailId}`);
        const prevIcon = document.getElementById(`toggle-icon-${openDetailId}`);
        const prevCard = document.getElementById(`endpoint-${openDetailId}`);

        if (prevDetail) {
            prevDetail.style.animation = 'slideDown 0.3s ease reverse forwards';
            setTimeout(() => {
                prevDetail.classList.add('hidden');
                prevDetail.style.animation = '';
            }, 300);
            prevIcon.textContent = '▼';
            prevCard.classList.remove('border-gray-400');
        }
    }

    if (detail.classList.contains('hidden')) {
        detail.classList.remove('hidden');
        detail.style.animation = 'slideDown 0.3s ease forwards';
        icon.textContent = '▲';
        card.classList.add('border-gray-400');
        openDetailId = id;

        setTimeout(() => {
            detail.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }, 100);
    } else {
        detail.style.animation = 'slideDown 0.3s ease reverse forwards';
        setTimeout(() => {
            detail.classList.add('hidden');
            detail.style.animation = '';
        }, 300);
        icon.textContent = '▼';
        card.classList.remove('border-gray-400');
        openDetailId = null;
    }
}

function renderEndpoints(endpoints) {
    const container = document.getElementById('endpoints-container');
    const startIndex = (currentPage - 1) * endpointsPerPage;
    const endIndex = startIndex + endpointsPerPage;
    const endpointsToShow = endpoints.slice(startIndex, endIndex);

    if (endpointsToShow.length === 0 && currentPage === 1) {
        container.innerHTML = '';
        return;
    }

    if (currentPage === 1) {
        container.innerHTML = endpointsToShow.map((api, index) => renderEndpointCard(api, index)).join('');
    } else {
        const newCards = endpointsToShow.map((api, index) => renderEndpointCard(api, startIndex + index)).join('');
        container.innerHTML += newCards;
    }

    endpointsToShow.forEach((api, index) => {
        const uniqueId = api.endpoint.replace(/[^a-zA-Z0-9]/g, '_') + '_' + (startIndex + index);
        setTimeout(() => {
            setupEndpointEvents(api, uniqueId);
        }, index * 50);
    });
}

function renderEndpointCard(api, index) {
    const uniqueId = api.endpoint.replace(/[^a-zA-Z0-9]/g, '_') + '_' + index;
    const methodStyle = getMethodStyle(api.method);
    const hasParams = api.parameter && api.parameter.length > 0;

    // Build parameter inputs
    const paramInputs = hasParams ? api.parameter.map(param => `
        <div class="space-y-1 sm:space-y-2">
            <div class="flex flex-col sm:flex-row sm:items-center justify-between gap-1 sm:gap-2">
                <label class="text-sm font-medium text-cyan-300">
                    ${param.nama}
                </label>
                <div class="flex items-center gap-2">
                    <span class="text-xs px-2 py-0.5 sm:py-1 rounded ${param.required ? 'bg-red-400/10 text-red-400 border border-red-400' : 'bg-gray-800 text-gray-400 border border-gray-700'}">
                        ${param.required ? 'required' : 'optional'}
                    </span>
                    <span class="text-xs text-gray-500">${param.tipe || 'string'}</span>
                </div>
            </div>
            <input type="${param.tipe === 'number' ? 'number' : 'text'}" 
                    class="param-input w-full px-3 py-2 border border-cyan-400/30 rounded-lg focus:ring-1 focus:ring-cyan-400 focus:border-cyan-400 outline-none text-sm bg-gray-900 text-gray-300"
                    data-param="${param.nama}"
                    data-type="${param.tipe || 'query'}"
                    placeholder="${param.contoh || (param.tipe === 'number' ? '123' : 'value')}"
                    ${param.required ? 'required' : ''}>
            ${param.deskripsi ? `<p class="text-xs text-gray-400 mt-1">${param.deskripsi}</p>` : ''}
        </div>
    `).join('') : `
        <div class="text-center py-3 sm:py-4 text-gray-400 text-sm">
            No parameters required for this endpoint.
        </div>
    `;

    // API Key input section
    const apiKeySection = `
        <div class="col-span-1 md:col-span-2">
            <div class="p-3 sm:p-4 bg-gray-900/50 rounded-lg border border-cyan-400/20 space-y-2">
                <div class="flex items-center justify-between">
                    <label class="text-sm font-medium text-cyan-300 flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 7a2 2 0 012 2m4 0a6 6 0 01-7.743 5.743L11 17H9v2H7v2H4a1 1 0 01-1-1v-2.586a1 1 0 01.293-.707l5.964-5.964A6 6 0 1121 9z" />
                        </svg>
                        API Key
                    </label>
                    <span class="text-xs px-2 py-0.5 sm:py-1 rounded bg-red-400/10 text-red-400 border border-red-400">
                        required
                    </span>
                </div>
                <div class="relative">
                    <input type="text" 
                            id="apikey-input-${uniqueId}"
                            class="param-input w-full px-3 py-2 border border-cyan-400/30 rounded-lg focus:ring-1 focus:ring-cyan-400 focus:border-cyan-400 outline-none text-sm bg-gray-900 text-gray-300"
                            data-param="apikey"
                            data-type="query"
                            placeholder="Enter your API key"
                            required
                            autocomplete="off">
                    <button type="button" 
                            onclick="copyToClipboard('apikey-input-${uniqueId}')"
                            class="absolute right-2 top-1/2 -translate-y-1/2 text-cyan-400 hover:text-cyan-300 p-1 rounded hover:bg-cyan-400/10">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                        </svg>
                    </button>
                </div>
                <p class="text-xs text-gray-400 mt-1">
                    Contact <a href="https://wa.me/6283154942624" target="_blank" class="text-cyan-400 hover:text-cyan-300 underline">wa.me/6283154942624</a> to get API key
                </p>
            </div>
        </div>
    `;

    // Build status table
    const statusTable = statusCodes.map((status, i) => `
        <tr class="hover:bg-gray-900">
            <td class="py-2 sm:py-3 px-3 sm:px-4">
                <div class="flex items-center gap-2">
                    <span class="font-mono font-bold ${status.color}">${status.icon}</span>
                    <code class="font-mono font-semibold text-gray-300">${status.code}</code>
                </div>
            </td>
            <td class="py-2 sm:py-3 px-3 sm:px-4 text-gray-400 text-sm">${status.desc}</td>
        </tr>
    `).join('');

    return `
        <div id="endpoint-${uniqueId}" class="endpoint-card rounded-lg sm:rounded-xl cyber-glow"
                style="animation-delay: ${index * 100}ms">
            
            <!-- Header -->
            <div class="p-4 sm:p-6 cursor-pointer group" onclick="toggleDetail('${uniqueId}')">
                <div class="flex flex-col sm:flex-row sm:items-start justify-between gap-3 sm:gap-0">
                    <div class="flex-1 min-w-0">
                        <div class="flex flex-wrap items-center gap-2 sm:gap-3 mb-2 sm:mb-3">
                            <span class="method-badge ${methodStyle.badge}">${api.method}</span>
                            <code class="text-sm font-mono text-cyan-300 wrap-url font-medium">${api.endpoint}</code>
                        </div>
                        <h3 class="text-base sm:text-lg font-semibold text-gray-300 mb-1 sm:mb-2 group-hover:text-cyan-300">${api.nama}</h3>
                        <p class="text-gray-400 text-sm sm:text-base">${api.deskripsi}</p>
                    </div>
                    <div class="flex items-center justify-between sm:justify-end gap-2 sm:gap-3">
                        <span class="px-2 sm:px-3 py-1 text-xs font-medium bg-gray-900 text-cyan-400 rounded-full border border-cyan-400">
                            ${api.kategori || 'General'}
                        </span>
                        <span id="toggle-icon-${uniqueId}" class="text-cyan-400 text-base sm:text-lg transition-transform">▼</span>
                    </div>
                </div>
            </div>
            
            <!-- Detail Panel -->
            <div id="detail-panel-${uniqueId}" class="hidden border-t border-gray-800">
                <div class="p-4 sm:p-6 space-y-6 sm:space-y-8">
                    
                    <!-- cURL Command -->
                    <section>
                        <div class="flex flex-col sm:flex-row sm:items-center justify-between gap-2 sm:gap-0 mb-3 sm:mb-4">
                            <h4 class="font-semibold text-cyan-300 text-base sm:text-lg">cURL Command</h4>
                            <button id="copy-curl-${uniqueId}" 
                                    class="text-xs sm:text-sm text-cyan-400 hover:text-cyan-300 flex items-center gap-1 sm:gap-2 px-2 sm:px-3 py-1 sm:py-1.5 rounded-lg hover:bg-cyan-400/10 transition-colors self-end sm:self-auto border border-cyan-400">
                                <svg class="w-3 h-3 sm:w-4 sm:h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                                </svg>
                                Copy
                            </button>
                        </div>
                        <div class="relative">
                            <pre id="curl-area-${uniqueId}" 
                                class="bg-gray-900 text-gray-100 rounded-lg p-3 sm:p-4 overflow-x-auto text-xs sm:text-sm font-mono leading-relaxed max-h-48 sm:max-h-96 border border-cyan-400/30">
${generateCurlCommand(api, {}, api.endpoint + (api.method === 'GET' ? '?apikey=YOUR_API_KEY' : ''))}
                            </pre>
                        </div>
                    </section>
                    
                    <!-- Tester -->
                    <section>
                        <div class="flex flex-col sm:flex-row sm:items-center justify-between gap-2 sm:gap-0 mb-4 sm:mb-6">
                            <h4 class="font-semibold text-cyan-300 text-base sm:text-lg">Try It Out</h4>
                            <span class="text-xs text-cyan-400 bg-cyan-400/10 px-2 sm:px-3 py-0.5 sm:py-1 rounded-full self-start sm:self-auto border border-cyan-400">Live Testing</span>
                        </div>
                        
                        <!-- Parameters Section including API Key -->
                        <div class="mb-6 sm:mb-8 p-4 sm:p-6 bg-gray-900 rounded-lg sm:rounded-xl border border-cyan-400/30 space-y-4 sm:space-y-6">
                            <h5 class="font-medium text-cyan-300 text-sm sm:text-base">Parameters</h5>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 sm:gap-6">
                                ${paramInputs}
                                ${apiKeySection}
                            </div>
                        </div>
                        
                        <button id="execute-btn-${uniqueId}" 
                                class="execute-btn w-full py-3 sm:py-4 rounded-lg sm:rounded-xl font-semibold text-sm sm:text-base mb-4 sm:mb-6">
                            <span id="execute-text-${uniqueId}">Execute ${api.method} Request</span>
                        </button>
                        
                        <div class="space-y-3 sm:space-y-4">
                            <div class="flex flex-col sm:flex-row sm:items-center justify-between gap-2 sm:gap-0">
                                <h5 class="font-semibold text-cyan-300 text-base sm:text-lg">Response</h5>
                                <div class="flex items-center gap-3 sm:gap-4">
                                    <span id="response-time-${uniqueId}" class="text-xs text-cyan-400"></span>
                                    <span id="response-status-${uniqueId}" class="text-xs font-medium"></span>
                                </div>
                            </div>
                            <div class="relative group">
                                <pre id="response-area-${uniqueId}" 
                                    class="bg-gray-900 text-gray-100 rounded-lg sm:rounded-xl p-3 sm:p-4 h-48 sm:h-80 overflow-auto text-xs sm:text-sm font-mono leading-relaxed border border-cyan-400/30">
// Response will appear here after executing request
                                </pre>
                                <button id="copy-response-${uniqueId}" 
                                        class="absolute top-2 sm:top-4 right-2 sm:right-4 text-cyan-400 hover:text-cyan-300 opacity-0 group-hover:opacity-100 transition-opacity p-1 sm:p-2 rounded hover:bg-cyan-400/10 border border-cyan-400">
                                    <svg class="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                                    </svg>
                                </button>
                                <div id="response-loader-${uniqueId}" class="hidden absolute inset-0 bg-gray-900/90 rounded-lg sm:rounded-xl flex items-center justify-center border border-cyan-400/30">
                                    <div class="loading-dots text-cyan-400">
                                        <span></span>
                                        <span></span>
                                        <span></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </section>
                    
                    <!-- Status Codes -->
                    <section>
                        <h4 class="font-semibold text-cyan-300 text-base sm:text-lg mb-4 sm:mb-6">Status Codes</h4>
                        <div class="table-container">
                            <div class="overflow-hidden rounded-lg sm:rounded-xl border border-cyan-400/30">
                                <table class="status-table w-full min-w-[300px]">
                                    <thead class="bg-gray-900">
                                        <tr>
                                            <th class="py-2 sm:py-3 px-3 sm:px-4 text-left text-xs sm:text-sm font-semibold text-cyan-300">Code</th>
                                            <th class="py-2 sm:py-3 px-3 sm:px-4 text-left text-xs sm:text-sm font-semibold text-cyan-300">Description</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-gray-800">
                                        ${statusTable}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </section>
                </div>
            </div>
        </div>
    `;
}

function setupEndpointEvents(api, id) {
    // Get all input elements
    const paramInputs = document.querySelectorAll(`#endpoint-${id} .param-input`);
    const apiKeyInput = document.getElementById(`apikey-input-${id}`);
    const curlArea = document.getElementById(`curl-area-${id}`);

    function updateCurl() {
        let url = api.endpoint;
        const params = {};

        // Get API key value
        if (apiKeyInput && apiKeyInput.value.trim()) {
            params['apikey'] = apiKeyInput.value.trim();
        }

        paramInputs.forEach(input => {
            const name = input.dataset.param;
            const type = input.dataset.type;
            const value = input.value.trim();

            if (value && name !== 'apikey') {
                if (api.method === 'GET' && type === 'query') {
                    params[name] = value;
                } else if (api.method !== 'GET') {
                    params[name] = value;
                }
            }
        });

        if (api.method === 'GET' && Object.keys(params).length > 0) {
            const query = new URLSearchParams(params).toString();
            url += '?' + query;
        }

        const bodyParams = api.method === 'GET' ? {} : params;
        curlArea.textContent = generateCurlCommand(api, bodyParams, url);
    }

    // Add event listeners to all inputs
    paramInputs.forEach(input => {
        input.addEventListener('input', updateCurl);
        input.addEventListener('focus', () => {
            input.classList.add('border-cyan-400');
        });
        input.addEventListener('blur', () => {
            input.classList.remove('border-cyan-400');
        });
    });

    if (apiKeyInput) {
        apiKeyInput.addEventListener('input', updateCurl);
    }

    // Copy cURL
    const copyCurlBtn = document.getElementById(`copy-curl-${id}`);
    if (copyCurlBtn) {
        copyCurlBtn.addEventListener('click', () => {
            navigator.clipboard.writeText(curlArea.textContent.trim()).then(() => {
                copyCurlBtn.classList.add('copy-success');
                copyCurlBtn.innerHTML = `
                    <svg class="w-3 h-3 sm:w-4 sm:h-4 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Copied
                `;
                setTimeout(() => {
                    copyCurlBtn.classList.remove('copy-success');
                    copyCurlBtn.innerHTML = `
                        <svg class="w-3 h-3 sm:w-4 sm:h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                        </svg>
                        Copy
                    `;
                }, 2000);
            });
        });
    }

    // Execute request
    const executeBtn = document.getElementById(`execute-btn-${id}`);
    const executeText = document.getElementById(`execute-text-${id}`);
    const responseArea = document.getElementById(`response-area-${id}`);
    const responseTime = document.getElementById(`response-time-${id}`);
    const responseStatus = document.getElementById(`response-status-${id}`);
    const responseLoader = document.getElementById(`response-loader-${id}`);

    if (executeBtn) {
        executeBtn.addEventListener('click', async () => {
            // Validate required params
            let isValid = true;
            const requiredInputs = Array.from(paramInputs).filter(input => input.required);
            const requiredApiKey = apiKeyInput && apiKeyInput.required;

            // Check API key
            if (requiredApiKey && (!apiKeyInput || !apiKeyInput.value.trim())) {
                isValid = false;
                if (apiKeyInput) {
                    apiKeyInput.classList.add('border-red-400');
                    setTimeout(() => {
                        apiKeyInput.classList.remove('border-red-400');
                    }, 2000);
                }
            }

            // Check other required params
            requiredInputs.forEach(input => {
                if (!input.value.trim()) {
                    isValid = false;
                    input.classList.add('border-red-400');
                    setTimeout(() => {
                        input.classList.remove('border-red-400');
                    }, 2000);
                }
            });

            if (!isValid) {
                executeBtn.style.background = 'linear-gradient(135deg, #b91c1c 0%, #ef4444 100%)';
                executeText.textContent = 'Required Parameters Missing!';
                setTimeout(() => {
                    executeBtn.style.background = '';
                    executeText.textContent = `Execute ${api.method} Request`;
                }, 2000);
                return;
            }

            // Build URL and body
            let url = api.endpoint;
            const params = {};
            const body = {};

            // Add API key
            if (apiKeyInput && apiKeyInput.value.trim()) {
                params['apikey'] = apiKeyInput.value.trim();
            }

            paramInputs.forEach(input => {
                const name = input.dataset.param;
                const type = input.dataset.type;
                const value = input.value.trim();

                if (value && name !== 'apikey') {
                    if (api.method === 'GET' && type === 'query') {
                        params[name] = value;
                    } else if (api.method !== 'GET') {
                        body[name] = value;
                    }
                }
            });

            if (api.method === 'GET' && Object.keys(params).length > 0) {
                const query = new URLSearchParams(params).toString();
                url += '?' + query;
            }

            // UI state
            executeBtn.disabled = true;
            executeText.innerHTML = `
                <div class="loading-dots">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
            `;

            if (responseLoader) {
                responseLoader.classList.remove('hidden');
            }

            responseArea.textContent = '';
            responseTime.textContent = '';
            responseStatus.textContent = '';

            const startTime = Date.now();

            try {
                const options = {
                    method: api.method,
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    }
                };

                // Add API key to headers for non-GET requests
                if (api.method !== 'GET' && apiKeyInput && apiKeyInput.value.trim()) {
                    options.headers['x-api-key'] = apiKeyInput.value.trim();
                }

                if (api.method !== 'GET' && Object.keys(body).length > 0) {
                    options.body = JSON.stringify(body);
                }

                const response = await fetch(url, options);
                const endTime = Date.now();
                const duration = endTime - startTime;

                let data;
                try {
                    data = await response.json();
                } catch {
                    const text = await response.text();
                    data = {
                        error: 'Invalid JSON response',
                        raw: text,
                        statusCode: response.status,
                        contentType: response.headers.get('content-type')
                    };
                }

                // Update UI
                responseArea.textContent = formatJSON(data);
                responseArea.style.animation = 'fadeInUp 0.3s ease';
                responseTime.textContent = `${duration}ms`;

                const statusColor = response.ok ? 'text-green-400' : 'text-red-400';
                const bgColor = response.ok ? 'bg-green-400/10' : 'bg-red-400/10';
                responseStatus.innerHTML = `
                    <span class="px-2 sm:px-3 py-0.5 sm:py-1 rounded-full ${bgColor} ${statusColor} font-semibold text-xs border ${response.ok ? 'border-green-400' : 'border-red-400'}">
                        ${response.status} ${response.statusText}
                    </span>
                `;

            } catch (error) {
                const endTime = Date.now();
                responseArea.textContent = `Error: ${error.message}`;
                responseTime.textContent = `${endTime - startTime}ms`;
                responseStatus.innerHTML = '<span class="px-2 sm:px-3 py-0.5 sm:py-1 rounded-full bg-red-400/10 text-red-400 font-semibold text-xs border border-red-400">Error</span>';
            } finally {
                executeBtn.disabled = false;
                executeText.textContent = `Execute ${api.method} Request`;

                if (responseLoader) {
                    responseLoader.classList.add('hidden');
                }
            }
        });
    }

    // Copy response
    const copyResponseBtn = document.getElementById(`copy-response-${id}`);
    if (copyResponseBtn) {
        copyResponseBtn.addEventListener('click', () => {
            navigator.clipboard.writeText(responseArea.textContent).then(() => {
                copyResponseBtn.innerHTML = `
                    <svg class="w-4 h-4 sm:w-5 sm:h-5 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                    </svg>
                `;
                copyResponseBtn.classList.add('copy-success');
                setTimeout(() => {
                    copyResponseBtn.innerHTML = `
                        <svg class="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                        </svg>
                    `;
                    copyResponseBtn.classList.remove('copy-success');
                }, 2000);
            });
        });
    }
}

// Initialize application
function initializeApp() {
    loadAPIData();

    // Search functionality with debounce
    const searchInput = document.getElementById('search-endpoints');
    let searchTimeout;

    searchInput.addEventListener('input', (e) => {
        clearTimeout(searchTimeout);

        searchTimeout = setTimeout(() => {
            const term = e.target.value.toLowerCase().trim();
            filterEndpoints(term);
        }, 300);
    });

    // Clear filters button
    document.getElementById('clear-filters').addEventListener('click', () => {
        activeCategory = 'all';
        currentSearchTerm = '';
        currentPage = 1;
        searchInput.value = '';

        document.querySelectorAll('.category-filter').forEach(btn =>
            btn.classList.remove('active-filter')
        );
        document.querySelector('[data-category="all"]')?.classList.add('active-filter');

        filterEndpoints();
    });

    // Reset search button
    document.getElementById('reset-search').addEventListener('click', () => {
        searchInput.value = '';
        filterEndpoints();
        document.getElementById('empty-state').classList.add('hidden');
        document.getElementById('endpoints-container').classList.remove('hidden');
    });

    // Show all button
    document.getElementById('show-all').addEventListener('click', () => {
        activeCategory = 'all';
        currentSearchTerm = '';
        searchInput.value = '';

        document.querySelectorAll('.category-filter').forEach(btn =>
            btn.classList.remove('active-filter')
        );
        document.querySelector('[data-category="all"]')?.classList.add('active-filter');

        filterEndpoints();
    });

    // Expand all button
    document.getElementById('expand-all').addEventListener('click', () => {
        document.querySelectorAll('[id^="detail-panel-"]').forEach(panel => {
            panel.classList.remove('hidden');
            panel.style.animation = 'slideDown 0.3s ease forwards';
        });
        document.querySelectorAll('[id^="toggle-icon-"]').forEach(icon => {
            icon.textContent = '▲';
        });
    });

    // Collapse all button
    document.getElementById('collapse-all').addEventListener('click', () => {
        document.querySelectorAll('[id^="detail-panel-"]').forEach(panel => {
            panel.style.animation = 'slideDown 0.3s ease reverse forwards';
            setTimeout(() => {
                panel.classList.add('hidden');
                panel.style.animation = '';
            }, 300);
        });
        document.querySelectorAll('[id^="toggle-icon-"]').forEach(icon => {
            icon.textContent = '▼';
        });
        openDetailId = null;
    });

    // Export JSON button
    document.getElementById('export-json').addEventListener('click', () => {
        const dataStr = JSON.stringify(allEndpoints, null, 2);
        const dataUri = 'data:application/json;charset=utf-8,' + encodeURIComponent(dataStr);

        const exportFileDefaultName = 'api-endpoints.json';

        const linkElement = document.createElement('a');
        linkElement.setAttribute('href', dataUri);
        linkElement.setAttribute('download', exportFileDefaultName);
        linkElement.click();
    });

    // Load more button
    document.getElementById('load-more').addEventListener('click', () => {
        currentPage++;
        const filtered = getFilteredEndpoints();
        renderEndpoints(filtered);
        updateLoadMoreButton(filtered.length);
    });

    // Close detail when clicking outside
    document.addEventListener('click', (e) => {
        if (openDetailId && !e.target.closest(`#endpoint-${openDetailId}`)) {
            toggleDetail(openDetailId);
        }
    });

    // Keyboard shortcuts
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && openDetailId) {
            toggleDetail(openDetailId);
        }
        if (e.key === '/' && e.target.tagName !== 'INPUT') {
            e.preventDefault();
            searchInput.focus();
        }
    });

    // FAB Functionality - NEW VERSION
    const fabSearchContainer = document.getElementById('fab-search-container');
    const fabSearchButton = document.getElementById('fab-search-button');
    const fabSearchField = document.getElementById('fab-search-field');
    const fabSearchClose = document.getElementById('fab-search-close');
    const backToTopFab = document.getElementById('back-to-top-fab');
    const mainSearchInput = document.getElementById('search-endpoints');

    // Open search field when FAB is clicked
    fabSearchButton.addEventListener('click', (e) => {
        e.stopPropagation();
        fabSearchContainer.classList.add('expanded');
        fabSearchField.focus();
    });

    // Close search field when close button is clicked
    fabSearchClose.addEventListener('click', (e) => {
        e.stopPropagation();
        closeFabSearch();
    });

    // Close search field when clicking outside
    document.addEventListener('click', (e) => {
        if (!fabSearchContainer.contains(e.target) && fabSearchContainer.classList.contains('expanded')) {
            closeFabSearch();
        }
    });

    // Handle search input in FAB with debounce
    let fabSearchTimeout;
    fabSearchField.addEventListener('input', (e) => {
        const term = e.target.value.trim();
        
        clearTimeout(fabSearchTimeout);
        fabSearchTimeout = setTimeout(() => {
            mainSearchInput.value = term;
            filterEndpoints(term);
            
            // Update search counter
            const filtered = getFilteredEndpoints();
            updateSearchCount(filtered.length);
        }, 300);
    });

    // Handle Enter and Escape keys in FAB search
    fabSearchField.addEventListener('keydown', (e) => {
        if (e.key === 'Enter') {
            e.preventDefault();
            fabSearchField.blur();
        }
        if (e.key === 'Escape') {
            closeFabSearch();
        }
    });

    // Sync main search input with FAB search
    mainSearchInput.addEventListener('input', (e) => {
        fabSearchField.value = e.target.value;
    });

    // Function to close FAB search
    function closeFabSearch() {
        fabSearchContainer.classList.remove('expanded');
        fabSearchField.value = '';
        
        // Clear main search if FAB search is empty
        if (!fabSearchField.value) {
            mainSearchInput.value = '';
            filterEndpoints('');
            updateSearchCount(getFilteredEndpoints().length);
        }
    }

    // Reset FAB search when clear filters is clicked
    document.getElementById('clear-filters').addEventListener('click', () => {
        fabSearchField.value = '';
    });

    // Reset FAB search when reset search is clicked
    document.getElementById('reset-search').addEventListener('click', () => {
        fabSearchField.value = '';
    });

    // Reset FAB search when show all is clicked
    document.getElementById('show-all').addEventListener('click', () => {
        fabSearchField.value = '';
    });

    // Initialize FAB search field with current search term
    fabSearchField.value = mainSearchInput.value;

    // Back to top FAB
    backToTopFab.addEventListener('click', () => {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });

    // Show/hide back to top FAB based on scroll position
    window.addEventListener('scroll', () => {
        if (window.scrollY > 300) {
            backToTopFab.style.opacity = '1';
            backToTopFab.style.pointerEvents = 'auto';
        } else {
            backToTopFab.style.opacity = '0.7';
            backToTopFab.style.pointerEvents = 'none';
        }
    });

    // Initialize back to top FAB visibility
    backToTopFab.style.opacity = '0.7';
    backToTopFab.style.pointerEvents = 'none';
}

// Make functions available globally for inline event handlers
window.copyToClipboard = copyToClipboard;
window.toggleDetail = toggleDetail;
window.loadAPIData = loadAPIData;

// Initialize the application when DOM is loaded
document.addEventListener('DOMContentLoaded', initializeApp);